<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Kegiatan;
use App\Models\KegiatanPartisipan;
use App\Models\Anggota;
use App\Models\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class KegiatanController extends Controller
{
    public function index(Request $request)
    {
        $query = Kegiatan::with([ 'partisipan']);
        
        // Filter berdasarkan jenis
        if ($request->filled('jenis')) {
            $query->where('jenis_kegiatan', $request->jenis);
        }
        
        // Filter berdasarkan status
        if ($request->filled('status')) {
            $query->where('status_kegiatan', $request->status);
        }
        
        // Filter berdasarkan tanggal
        if ($request->filled('tanggal_dari')) {
            $query->whereDate('tanggal_kegiatan', '>=', $request->tanggal_dari);
        }
        
        if ($request->filled('tanggal_sampai')) {
            $query->whereDate('tanggal_kegiatan', '<=', $request->tanggal_sampai);
        }
        
        // Search berdasarkan nama kegiatan
        if ($request->filled('search')) {
            $query->where('nama_kegiatan', 'like', '%' . $request->search . '%');
        }
        
        $kegiatans = $query->orderBy('tanggal_kegiatan', 'desc')->paginate(15);
        
        return view('kegiatan.index', compact('kegiatans'));
    }

    public function create()
    {
        $anggotas = Anggota::where('status_keanggotaan', 'Aktif')
                          ->orderBy('nama_lengkap')
                          ->get();
        $users = User::orderBy('name')->get();
        
        return view('kegiatan.create', compact('anggotas', 'users'));
    }

    public function store(Request $request)
    {
        $rules = [
            'nama_kegiatan' => 'required|string|max:255',
            'deskripsi' => 'nullable|string',
            'tanggal_kegiatan' => 'required|date',
            'waktu_mulai' => 'required',
            'waktu_selesai' => 'nullable|after:waktu_mulai',
            'tempat' => 'required|string|max:500',
            'jenis_kegiatan' => 'required|in:Mabit,Rapat,Pelatihan,Sosialisasi,Lainnya',
            'target_peserta' => 'nullable|integer|min:1',
            'catatan' => 'nullable|string',
            'pic_id' => 'nullable|exists:users,id',
            'partisipan' => 'nullable|array',
            'partisipan.*' => 'exists:tbl_anggota,id',
            'status_partisipasi' => 'nullable|array',
            'status_partisipasi.*' => 'in:Wajib,Opsional,Undangan',
        ];
        
        $messages = [
            'nama_kegiatan.required' => 'Nama kegiatan wajib diisi.',
            'tanggal_kegiatan.required' => 'Tanggal kegiatan wajib diisi.',
            'waktu_mulai.required' => 'Waktu mulai wajib diisi.',
            'waktu_selesai.after' => 'Waktu selesai harus setelah waktu mulai.',
            'tempat.required' => 'Tempat kegiatan wajib diisi.',
            'jenis_kegiatan.required' => 'Jenis kegiatan wajib dipilih.',
            'target_peserta.min' => 'Target peserta minimal 1 orang.',
            'partisipan.*.exists' => 'Anggota yang dipilih tidak valid.',
        ];

        $request->validate($rules, $messages);

        try {
            DB::beginTransaction();

            // Create kegiatan
            $kegiatan = Kegiatan::create($request->only([
                'nama_kegiatan', 'deskripsi', 'tanggal_kegiatan', 'waktu_mulai', 
                'waktu_selesai', 'tempat', 'jenis_kegiatan', 'target_peserta', 
                'catatan', 'pic_id'
            ]));

            // Add partisipan jika ada
            if ($request->filled('partisipan')) {
                foreach ($request->partisipan as $index => $anggotaId) {
                    $statusPartisipasi = $request->status_partisipasi[$index] ?? 'Wajib';
                    
                    KegiatanPartisipan::create([
                        'kegiatan_id' => $kegiatan->id,
                        'anggota_id' => $anggotaId,
                        'status_partisipasi' => $statusPartisipasi,
                        'status_kehadiran' => 'Belum Konfirmasi'
                    ]);
                }
            }

            DB::commit();
            
            return redirect()->route('kegiatan.show', $kegiatan->id)
                           ->with('success', 'Kegiatan berhasil dibuat!');
                           
        } catch (\Exception $e) {
            DB::rollback();
            
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Gagal membuat kegiatan: ' . $e->getMessage());
        }
    }

    public function show(Kegiatan $kegiatan)
    {
        $kegiatan->load(['pic', 'partisipan.anggota']);
        
        return view('kegiatan.show', compact('kegiatan'));
    }

    public function edit(Kegiatan $kegiatan)
    {
        if (!$kegiatan->canBeEdited()) {
            return redirect()->route('kegiatan.show', $kegiatan)
                           ->with('error', 'Kegiatan yang sudah selesai tidak dapat diedit!');
        }
        
        $kegiatan->load(['partisipan']);
        $anggotas = Anggota::where('status_keanggotaan', 'Aktif')
                          ->orderBy('nama_lengkap')
                          ->get();
        $users = User::orderBy('name')->get();
        
        return view('kegiatan.edit', compact('kegiatan', 'anggotas', 'users'));
    }

    public function update(Request $request, Kegiatan $kegiatan)
    {
        if (!$kegiatan->canBeEdited()) {
            return redirect()->route('kegiatan.show', $kegiatan)
                           ->with('error', 'Kegiatan yang sudah selesai tidak dapat diedit!');
        }

        $rules = [
            'nama_kegiatan' => 'required|string|max:255',
            'deskripsi' => 'nullable|string',
            'tanggal_kegiatan' => 'required|date',
            'waktu_mulai' => 'required',
            'waktu_selesai' => 'nullable|after:waktu_mulai',
            'tempat' => 'required|string|max:500',
            'jenis_kegiatan' => 'required|in:Mabit,Rapat,Pelatihan,Sosialisasi,Lainnya',
            'status_kegiatan' => 'required|in:Direncanakan,Berlangsung,Selesai,Dibatalkan',
            'target_peserta' => 'nullable|integer|min:1',
            'catatan' => 'nullable|string',
            'pic_id' => 'nullable|exists:users,id',
            'partisipan' => 'nullable|array',
            'partisipan.*' => 'exists:tbl_anggota,id',
            'status_partisipasi' => 'nullable|array',
            'status_partisipasi.*' => 'in:Wajib,Opsional,Undangan',
        ];

        $request->validate($rules);

        try {
            DB::beginTransaction();

            // Update kegiatan
            $kegiatan->update($request->only([
                'nama_kegiatan', 'deskripsi', 'tanggal_kegiatan', 'waktu_mulai', 
                'waktu_selesai', 'tempat', 'jenis_kegiatan', 'status_kegiatan',
                'target_peserta', 'catatan', 'pic_id'
            ]));

            // Update partisipan
            if ($request->filled('partisipan')) {
                // Hapus partisipan lama
                $kegiatan->partisipan()->delete();
                
                // Tambah partisipan baru
                foreach ($request->partisipan as $index => $anggotaId) {
                    $statusPartisipasi = $request->status_partisipasi[$index] ?? 'Wajib';
                    
                    KegiatanPartisipan::create([
                        'kegiatan_id' => $kegiatan->id,
                        'anggota_id' => $anggotaId,
                        'status_partisipasi' => $statusPartisipasi,
                        'status_kehadiran' => 'Belum Konfirmasi'
                    ]);
                }
            } else {
                // Jika tidak ada partisipan yang dipilih, hapus semua
                $kegiatan->partisipan()->delete();
            }

            DB::commit();
            
            return redirect()->route('kegiatan.show', $kegiatan)
                           ->with('success', 'Kegiatan berhasil diupdate!');
                           
        } catch (\Exception $e) {
            DB::rollback();
            
            return redirect()->back()
                           ->withInput()
                           ->with('error', 'Gagal update kegiatan: ' . $e->getMessage());
        }
    }

    public function destroy(Kegiatan $kegiatan)
    {
        if (!$kegiatan->canBeEdited()) {
            return redirect()->route('kegiatan.index')
                           ->with('error', 'Kegiatan yang sudah selesai tidak dapat dihapus!');
        }

        try {
            DB::beginTransaction();
            
            // Hapus partisipan dulu (karena foreign key)
            $kegiatan->partisipan()->delete();
            
            // Hapus kegiatan
            $kegiatan->delete();
            
            DB::commit();
            
            return redirect()->route('kegiatan.index')
                           ->with('success', 'Kegiatan berhasil dihapus!');
                           
        } catch (\Exception $e) {
            DB::rollback();
            
            return redirect()->route('kegiatan.index')
                           ->with('error', 'Gagal menghapus kegiatan: ' . $e->getMessage());
        }
    }

    public function updateKehadiran(Request $request, $id)
    {
        $partisipan = KegiatanPartisipan::findOrFail($id);
        
        $request->validate([
            'status_kehadiran' => 'required|in:Belum Konfirmasi,Hadir,Tidak Hadir,Izin',
            'keterangan' => 'nullable|string|max:255'
        ]);

        $partisipan->update([
            'status_kehadiran' => $request->status_kehadiran,
            'keterangan' => $request->keterangan,
            'waktu_konfirmasi' => now()
        ]);

        return redirect()->back()
                       ->with('success', 'Status kehadiran berhasil diupdate!');
    }
}